local Image = require "widgets/image"
local ImageButton = require "widgets/imagebutton"
local Text = require "widgets/text"
local Widget = require "widgets/widget"
local UIAnim = require "widgets.uianim"

local FrenzySelectionWidget = require"widgets.ftf.dungeonselection.frenzyselectionwidget"
local LocationBossesWidget = require"widgets.ftf.dungeonselection.locationbosseswidget"
local DungeonLevelWidget = require"widgets/ftf/dungeonlevelwidget"

local MetaProgress = require "defs.metaprogression"

local playerutil = require"util/playerutil"
local easing = require "util.easing"

local monster_pictures = require "gen.atlas.monster_pictures"

local EquipmentStatDisplay = require "defs.equipmentstatdisplay"

------------------------------------------------------------------------------------------

-- If the dungeon's value is 0.4, we will modify the value we show the player to make a better experience.
-- NOTE: Light armor of a given dungeon gives -0.05.
local RECOMMENDED_ATTACK_BUFFER_DEFAULT = 0.00
local RECOMMENDED_DEFEND_BUFFER_DEFAULT = 0.00

-- If the player's current score is this % less than the recommended value, we will show red or yellow.
local WARNING_THRESHOLD_YELLOW = 0.75
local WARNING_THRESHOLD_RED = 0.5
local WARNING_THRESHOLD_SERIOUSLYDONT = 0.35

-- Because the numbers in early game are so low, the same assessments are not true.
local EARLYGAME_WARNING_THRESHOLD_YELLOW = 0.5
local EARLYGAME_WARNING_THRESHOLD_RED = nil
local EARLYGAME_WARNING_THRESHOLD_SERIOUSLYDONT = nil

local RED_COLOR = RGB(252, 81, 81)
local YELLOW_COLOR = RGB(214, 186, 74)

-- The normal difficulty assessment just calculates the basic 'additional damage mod' and 'additional health' of mobs
-- However, ascensions add much more than just that, so we should add some more to these values to represent things like "decreased cooldowns" and additional boss health, etc.
local RECOMMENDED_ATTACK_BUFFER_MOD_ASCENSION =
{
	0.0,
	0.05,
	0.1,
	0.2,
}
local RECOMMENDED_DEFEND_BUFFER_MOD_ASCENSION =
{
	0.0,
	0.05, --0.1,
	0.1, --0.2,
	0.2, --0.3,
}

--- A map info panel
----
local MapSidebar = Class(Widget, function(self)
	Widget._ctor(self, "MapSidebar")

	self.width = 1270
	self.height = 1630

	self.bosses_width = self.width + 100

	-- Background
	self.bg = self:AddChild(Image("images/map_ftf/panel_bg.tex"))
		:SetName("Background")

	-- The boss header backing, behind the monsters and the background
	self.boss_header_hitbox = self:AddChild(Image("images/global/square.tex"))
		:SetName("Boss header hitbox")
		:SetSize(self.width, 320)
		:SetMultColor(UICOLORS.DEBUG)
		:SetMultColorAlpha(0)
	self.boss_header_anim_back = self:AddChild(UIAnim())
		:SetName("Bosses header anim")
		:SetScale(1.17)
		:SetHiddenBoundingBox(true)
		:SendToBack()
	self.boss_header_anim_back:GetAnimState():SetBank("world_map_banner")
	self.boss_header_anim_back:GetAnimState():SetBuild("world_map_banner")
	self.boss_header_anim_back:GetAnimState():PlayAnimation("treemon_forest_back", false)
	-- The monsters
	self.bosses_widget = self:AddChild(LocationBossesWidget(self.bosses_width, nil))
		:SetName("Bosses widget")
		:SetHiddenBoundingBox(true)
	-- The foreground, in front of the monsters and background
	self.boss_header_anim_front = self:AddChild(UIAnim())
		:SetName("Bosses header anim")
		:SetScale(1.17)
		:SetHiddenBoundingBox(true)
	self.boss_header_anim_front:GetAnimState():SetBank("world_map_banner")
	self.boss_header_anim_front:GetAnimState():SetBuild("world_map_banner")
	self.boss_header_anim_front:GetAnimState():PlayAnimation("treemon_forest_front", false)

	-- Location title and exploration progress bar
	self.dungeon_level = self:AddChild(DungeonLevelWidget())
		:SetName("Dungeon level widget")
		:ShowLargePresentation(HexToRGB(0xA3897B77), UICOLORS.LIGHT_TEXT, FONTSIZE.SCREEN_TITLE, self.width)
		:SetHiddenBoundingBox(true)

	-- Location stats
	self.attack_label = self:AddChild(Text(FONTFACE.DEFAULT, FONTSIZE.SCREEN_TEXT, STRINGS.UI.MAPSCREEN.RECOMMENDED_ATTACK_LABEL, UICOLORS.LIGHT_TEXT_DARK))
		:SetName("Attack label")
		-- :Hide()
	self.attack_value = self:AddChild(Text(FONTFACE.DEFAULT, FONTSIZE.SCREEN_TEXT*2.5, STRINGS.UI.MAPSCREEN.ATTACK_VALUE, UICOLORS.LIGHT_TEXT))
		:SetName("Attack value")
	self.attack_current = self:AddChild(Text(FONTFACE.DEFAULT, FONTSIZE.SCREEN_TEXT*1.2, STRINGS.UI.MAPSCREEN.CURRENT_ATTACK, UICOLORS.LIGHT_TEXT_DARK))
		:SetName("Attack current")
	self.attack_warning = self:AddChild(Image("images/ui_ftf/warning.tex"))
		:SetScale(0.35, 0.35)
		:SetName("Attack warning")
		:Hide() -- Only show these conditionally

	self.defense_label = self:AddChild(Text(FONTFACE.DEFAULT, FONTSIZE.SCREEN_TEXT, STRINGS.UI.MAPSCREEN.RECOMMENDED_DEFENSE_LABEL, UICOLORS.LIGHT_TEXT_DARK))
		:SetName("Defense label")
		-- :Hide()
	self.defense_value = self:AddChild(Text(FONTFACE.DEFAULT, FONTSIZE.SCREEN_TEXT*2.5, STRINGS.UI.MAPSCREEN.DEFENSE_VALUE, UICOLORS.LIGHT_TEXT))
		:SetName("Defense value")
	self.defense_current = self:AddChild(Text(FONTFACE.DEFAULT, FONTSIZE.SCREEN_TEXT*1.2, STRINGS.UI.MAPSCREEN.CURRENT_DEFENSE, UICOLORS.LIGHT_TEXT_DARK))
		:SetName("Defense current")
	self.defense_warning = self:AddChild(Image("images/ui_ftf/warning.tex"))
		:SetScale(0.35, 0.35)
		:SetName("Defense warning")
		:Hide() -- Only show these conditionally

	-- Ascension selector widget
	self.ascension_widget = self:AddChild(FrenzySelectionWidget())
		:SetName("Ascension widget")
		:SetOnSelectLevelFn(function()
			-- self:RefreshMaterialDrops()
			self:UpdateDifficultyAssessment()
			self:DoLayout()
		end)

	self.stats_container = self:AddChild(Widget())
		:SetName("Stats container")

	-- Mobs list
	self.mobs_title = self:AddChild(Text(FONTFACE.DEFAULT, FONTSIZE.SCREEN_TEXT*1.3, STRINGS.ASCENSIONS.MOBS_TITLE, UICOLORS.LIGHT_TEXT))
		:SetName("Mobs title")
	self.mobs_container = self:AddChild(Widget())
		:SetName("Mobs container")

	-- Close button
	self.close_button = self:AddChild(ImageButton("images/ui_ftf/HeaderClose.tex"))
		:SetNavFocusable(false) -- rely on CONTROL_MAP
		:SetSize(BUTTON_SQUARE_SIZE, BUTTON_SQUARE_SIZE)
		:LayoutBounds("right", "top", self.bg)
		:Offset(20, 0)
		:SetOnClick(function()
			if self.on_close_fn then self.on_close_fn() end
		end)
end)

function MapSidebar:SetPlayer(player)
	self:SetOwningPlayer(player)

	self.dungeon_level:SetPlayer(player)
	self.ascension_widget:SetPlayer(player)

	self:UpdateDifficultyAssessment(player)

	return self
end

function MapSidebar:IsUnlocked(locationData)
	for i,key in ipairs(locationData.required_unlocks) do
		if not self:GetOwningPlayer().components.unlocktracker:IsLocationUnlocked(key) then -- Region unlock check
			return false
		end
	end
	return true
end

function MapSidebar:SetLocationData(locationData)
	-- Save data
	self.locationData = locationData

	-- Update header
	local location_id = self.locationData.id
	self.boss_header_anim_back:GetAnimState():PlayAnimation(location_id .. "_back", false)
	self.boss_header_anim_front:GetAnimState():PlayAnimation(location_id .. "_front", false)

	-- Refresh level widget
	self.dungeon_level:SetBiomeTitle(self.locationData.pretty.name)
	self.dungeon_level:ShouldPlaySound(false)
	local mrm = self:GetOwningPlayer().components.metaprogressmanager
	local def = MetaProgress.FindProgressByName(self.locationData.id)
	local progress = mrm:GetProgress(def)
	if not progress and def ~= nil then
		-- The player hasn't made progress on this location
		progress = mrm:StartTrackingProgress(mrm:CreateProgress(def))
	end

	local meta_progress = {
		meta_reward = progress,
		meta_reward_def = progress.def,
		meta_level = progress:GetLevel(),
		meta_exp = progress:GetEXP(),
		meta_exp_max = progress:GetEXPForLevel(progress:GetLevel()),
	}

	self.dungeon_level:RefreshMetaProgress(meta_progress)

	-- self.dungeon_level:SetProgress(progress:GetLevel(), progress:GetEXP(), progress:GetEXPForLevel(progress:GetLevel()))

	-- Update ascension widget
	self.ascension_widget:SetLocation(self.locationData)

	self:UpdateDifficultyAssessment()

	-- Update center contents
	self:RefreshContents()
	self:StartUpdating()

	return self
end
function MapSidebar:UpdateDifficultyAssessment()
	local scene_gen = self.locationData:GetSceneGen()
	scene_gen = SpawnPrefab(scene_gen)
	local dungeon_tier = scene_gen.components.scenegen:GetTier()
	scene_gen:Remove()

	local frenzy = self.ascension_widget.selected_level

	local enemy_modifiers = TUNING:GetEnemyModifiersAtAscensionAndTier(nil, frenzy, dungeon_tier)

	-- How much extra damage the dungeon/frenzy does. Match against player_def.
	local dungeon_atk = enemy_modifiers.DungeonTierDamageMult
	dungeon_atk = dungeon_atk + RECOMMENDED_ATTACK_BUFFER_DEFAULT
	dungeon_atk = dungeon_atk + RECOMMENDED_ATTACK_BUFFER_MOD_ASCENSION[frenzy+1]

	-- How much extra health the dungeon/frenzy imbues. Match against player_atk.
	local dungeon_def = enemy_modifiers.HealthMult + enemy_modifiers.BasicHealthMult
	dungeon_def = dungeon_def + RECOMMENDED_DEFEND_BUFFER_DEFAULT
	dungeon_def = dungeon_def + RECOMMENDED_DEFEND_BUFFER_MOD_ASCENSION[frenzy+1]

	printf("[UpdateDifficultyAssessment] Recommended DEF: %s + buffer %s + frenzy %s = %s. Recommended ATK: %s + buffer %s + frenzy %s = %s", enemy_modifiers.DungeonTierDamageMult, RECOMMENDED_ATTACK_BUFFER_DEFAULT, RECOMMENDED_ATTACK_BUFFER_MOD_ASCENSION[frenzy+1], dungeon_atk, enemy_modifiers.HealthMult + enemy_modifiers.BasicHealthMult, RECOMMENDED_DEFEND_BUFFER_DEFAULT, RECOMMENDED_DEFEND_BUFFER_MOD_ASCENSION[frenzy+1], dungeon_def)

	local armour_display = EquipmentStatDisplay[EQUIPMENT_STATS.s.ARMOUR]
	local dmg_display = EquipmentStatDisplay[EQUIPMENT_STATS.s.DMG_MULT]

	local player = self:GetOwningPlayer()
	local player_stats = player.components.inventoryhoard:ComputeStats()
	local player_def = player_stats.ARMOUR
	local player_atk = player_stats.DMG_MULT

	-- Calculate display values
	local player_atk_display = dmg_display.displayvalue_fn(EQUIPMENT_STATS.s.DMG_MULT, player_atk)
	local player_def_display = armour_display.displayvalue_fn(EQUIPMENT_STATS.s.ARMOUR, player_def)
	local dungeon_atk_display = armour_display.displayvalue_fn(EQUIPMENT_STATS.s.ARMOUR, dungeon_atk)
	local dungeon_def_display = dmg_display.displayvalue_fn(EQUIPMENT_STATS.s.DMG_MULT, dungeon_def)

	-- Update widgets
	self.attack_value:SetText(string.format(STRINGS.UI.MAPSCREEN.ATTACK_VALUE, math.floor(dungeon_def_display)))
	self.defense_value:SetText(string.format(STRINGS.UI.MAPSCREEN.DEFENSE_VALUE, math.floor(dungeon_atk_display)))
	self.attack_current:SetText(string.format(STRINGS.UI.MAPSCREEN.CURRENT_ATTACK, math.floor(player_atk_display)))
	self.defense_current:SetText(string.format(STRINGS.UI.MAPSCREEN.CURRENT_DEFENSE, math.floor(player_def_display)))

	local atk_percent = player_atk_display / dungeon_def_display
	local atk_color = UICOLORS.LIGHT_TEXT_DARK
	local def_percent = player_def_display / dungeon_atk_display
	local def_color = UICOLORS.LIGHT_TEXT_DARK

	local earlygame = dungeon_atk_display <= 20 or dungeon_def_display <= 20

	local warn_threshold
	local red_threshold
	local yellow_threshold
	if earlygame then
		warn_threshold = EARLYGAME_WARNING_THRESHOLD_SERIOUSLYDONT
		red_threshold = EARLYGAME_WARNING_THRESHOLD_RED
		yellow_threshold = EARLYGAME_WARNING_THRESHOLD_YELLOW
	else
		warn_threshold = WARNING_THRESHOLD_SERIOUSLYDONT
		red_threshold = WARNING_THRESHOLD_RED
		yellow_threshold = WARNING_THRESHOLD_YELLOW
	end

	if warn_threshold ~= nil and atk_percent < warn_threshold then
		atk_color = RED_COLOR
		self.attack_warning:Show()
	elseif red_threshold ~= nil and atk_percent < red_threshold then
		atk_color = RED_COLOR
		self.attack_warning:Hide()
	elseif yellow_threshold ~= nil and atk_percent < yellow_threshold then
		atk_color = YELLOW_COLOR
		self.attack_warning:Hide()
	else
		self.attack_warning:Hide()
	end

	if warn_threshold ~= nil and def_percent < warn_threshold then
		def_color = RED_COLOR
		self.defense_warning:Show()
	elseif red_threshold ~= nil and def_percent < red_threshold then
		def_color = RED_COLOR
		self.defense_warning:Hide()
	elseif yellow_threshold ~= nil and def_percent < yellow_threshold then
		def_color = YELLOW_COLOR
		self.defense_warning:Hide()
	else
		self.defense_warning:Hide()
	end

	-- self.attack_value:SetGlyphColor(warn_atk and UICOLORS.RED or UICOLORS.LIGHT_TEXT)
	-- self.defense_value:SetGlyphColor(warn_def and UICOLORS.RED or UICOLORS.LIGHT_TEXT)

	self.attack_current:SetGlyphColor(atk_color)
	self.defense_current:SetGlyphColor(def_color)
end

function MapSidebar:OnUpdate()
	local is_unlocked = playerutil.GetLocationUnlockInfo(self.locationData)

	if is_unlocked ~= self.is_unlocked then
		self:RefreshContents()
	else
		self.ascension_widget:OnUpdate()
	end
end

function MapSidebar:RefreshContents()

	-- Refresh monsters
	self.bosses_widget:SetMonsters(self.locationData.monsters)

	if self:GetOwningPlayer() then
		-- when the screen refreshes without calling set player again
		-- happens when closing the 'change weapon' overlay screen
		self.ascension_widget:SetPlayer(self:GetOwningPlayer())
	end

	-- Update local mobs list
	self:UpdateMobs()

	-- Update footer
	local is_unlocked, invalid_players = playerutil.GetLocationUnlockInfo(self.locationData)

	self.is_unlocked = is_unlocked
	self.invalid_players = invalid_players

	self:DoLayout()
end

function MapSidebar:UpdateMobs()
	-- Remove old ones
	self.mobs_container:RemoveAllChildren()

	-- Add new ones
	local monster_count = 1
	for k, monster_id in ipairs(self.locationData.monsters.mobs) do

		-- Only show named creatures
		local monster_name = STRINGS.NAMES[monster_id]
		if monster_name then

			local even = monster_count%2 == 0
			local unlocked = self:GetOwningPlayer().components.unlocktracker:IsEnemyUnlocked(monster_id)

			local widget = self.mobs_container:AddChild(Widget())
				:SetName("Mob "..monster_count)

			local texture = even and "images/map_ftf/rots_bg_even.tex" or "images/map_ftf/rots_bg_odd.tex"
			widget.bg = widget:AddChild(Image(texture))
				:SetName("Bg")
				:SetMultColor(HexToRGB(0x302423FF))
			widget.icon = widget:AddChild(Image(monster_pictures.tex[string.format("research_widget_%s", monster_id)]))
				:SetName("Icon")
				:SetHiddenBoundingBox(true)
				:SetScale(0.45)

			if unlocked then

				-- Show a tooltip with the creature name, if unlocked
				widget:SetToolTip(STRINGS.NAMES[monster_id])
			else

				-- Make this monster a silhouette if not seen before
				widget.icon
					:SetMultColor(UICOLORS.BLACK)
					:SetAddColor(HexToRGB(0x181312FF))
				widget:SetToolTip(STRINGS.UI.MAPSCREEN.UNKNOWN_CREATURE)
			end

			monster_count = monster_count + 1
		end

	end
end

function MapSidebar:DoLayout()

	self.boss_header_hitbox:LayoutBounds("center", "above", self.bg)
		:Offset(0, -10)
	self.boss_header_anim_back:LayoutBounds("center", "above", self.bg)
		:Offset(-20, -10)
	self.bosses_widget:LayoutBounds("center", "bottom", self.boss_header_anim_back)
		:Offset(30, -40)
	self.boss_header_anim_front:SetPos(self.boss_header_anim_back:GetPos())

	self.dungeon_level:LayoutBounds("center", "above", self.bg)
		:Offset(0, -450)

	-- Layout attack and defense
	self.attack_label:LayoutBounds("center", "top", self.bg):Offset(-385, -270)
	self.defense_label:LayoutBounds("center", "top", self.bg):Offset(385, -270)
	self.attack_value:LayoutBounds("center", "below", self.attack_label):Offset(0, 0)
	self.defense_value:LayoutBounds("center", "below", self.defense_label):Offset(0, 0)
	self.attack_current:LayoutBounds("center", "below", self.attack_label):Offset(0, -125)
	self.defense_current:LayoutBounds("center", "below", self.defense_label):Offset(0, -125)

	self.attack_warning:LayoutBounds("after", "center", self.attack_current):Offset(10, 0)
	self.defense_warning:LayoutBounds("after", "center", self.defense_current):Offset(10, 0)

	self.ascension_widget:SetPos(0, 7)

	self.mobs_title:LayoutBounds("center", "center", self.bg)
		:Offset(0, -335)
	self.mobs_container:LayoutChildrenInGrid(6, {h=5, v=10})
		:LayoutBounds("center", "center", self.bg)
		:Offset(0, -565)

	return self
end

function MapSidebar:SetOnCloseFn(fn)
	self.on_close_fn = fn
	return self
end

function MapSidebar:IncreaseAscensionLevel()
	self.ascension_widget:DeltaLevel(1)
	return self
end

function MapSidebar:DecreaseAscensionLevel()
	self.ascension_widget:DeltaLevel(-1)
	return self
end

function MapSidebar:SetOnLocationUnlockedFn(fn)
	self.onLocationUnlockedFn = fn
	return self
end

function MapSidebar:PrepareAnimation()
	self:SetMultColorAlpha(0)
	return self
end

function MapSidebar:AnimateIn(on_done)

	local target_x, target_y = self:GetPos()
	self:Offset(0, -70)
	self:MoveTo(target_x, target_y, 0.6, easing.outElasticUI)
	self:AlphaTo(1, 0.2, easing.outQuad, on_done)

	return self
end

return MapSidebar
